﻿using System;

namespace Microscopic_Traffic_Simulator___Model.CellularTopologyObjects.GeneralParameters
{
    /// <summary>
    /// Class containing parameters related to cellular topology.
    /// </summary>
    [Serializable]
    public class CellularTopologyParameters
    {
        /// <summary>
        /// Cell length parameter.
        /// </summary>
        public double P1_CellLength { get; set; }

        /// <summary>
        /// Cell length parameter.
        /// </summary>
        public TimeSpan P2_SimulationStepInterval { get; set; }

        /// <summary>
        /// Value of the number of meters in a kilometer.
        /// </summary>
        private const double MetersInKilometer = 1000.0;

        /// <summary>
        /// Value of the number of seconds in an hour.
        /// </summary>
        private const double SecondsInHour = 3600.0;

        /// <summary>
        /// Converts speed in KPH to speed in number of cells per simulation step.
        /// </summary>
        /// <param name="speed">Speed in KPH.</param>
        /// <returns>Speed in number of cells per simulation step.</returns>        
        internal int FromKphSpeedToSimSpeed(double kph)
        {
            return (int)Math.Round(kph *
                (MetersInKilometer * P2_SimulationStepInterval.TotalSeconds) /
                (SecondsInHour * P1_CellLength));
        }

        /// <summary>
        /// Converts simulation speed to speed in kph.
        /// </summary>
        /// <param name="simSpeed">Simulation speed (in number of cells per simulation speed).</param>
        /// <returns>Speed in KPH.</returns>
        internal double FromSimSpeedToKphSpeed(int simSpeed)
        {
            return simSpeed *
                (SecondsInHour * P1_CellLength) /
                (MetersInKilometer * P2_SimulationStepInterval.TotalSeconds);
        }
    }
}
